package com.cftech.addresst.web;

import com.alibaba.fastjson.JSONObject;
import com.cftech.addresst.model.Address;
import com.cftech.addresst.model.Area;
import com.cftech.addresst.service.AddressService;
import com.cftech.core.poi.ExcelKit;
import com.cftech.core.scope.OrderType;
import com.cftech.core.sql.Conds;
import com.cftech.core.sql.Sort;
import com.cftech.core.util.Constants;
import com.cftech.sys.security.PermissionSign;
import com.cftech.sys.security.UserUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.shiro.authz.annotation.RequiresPermissions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.File;
import java.io.IOException;
import java.util.List;


/**
 * 收货地址管理Controller
 * <p>
 * 权限字符串说明:
 * 查看:public static final String ADDRESS_VIEW = "qy:address:view"
 * 查看:public static final String ADDRESS_EDIT = "qy:address:edit"
 *
 * @author Licc
 * @date: 2020-09-29 17:17
 */
@Slf4j
@Controller
@RequestMapping("/a/address")
public class AddressController {

    public static final String ADDRESS_VIEW = "qy:address:view";
    public static final String ADDRESS_EDIT = "qy:address:edit";

    @Autowired
    private AddressService addressService;


    //列表页面
    @RequiresPermissions(value = ADDRESS_VIEW)
    @RequestMapping("/list")
    public String list(HttpServletRequest request, Model model) {
        Long accountId = UserUtils.getmpaccounts(request);
        model.addAttribute("accountId", accountId);
        return "address/addresslist";
    }

    //编辑页面（新增、修改）
    @RequiresPermissions(value = ADDRESS_VIEW)
    @RequestMapping("/form")
    public String form(HttpServletRequest request, String id, Model model) {
        if (!StringUtils.isEmpty(id)) {
            Address address = addressService.fetchById(id);
            model.addAttribute("data", address);
        }
        return "address/addressform";
    }

    //提交数据（新增、修改）
    @RequiresPermissions(value = ADDRESS_EDIT)
    @RequestMapping(value = "/formData",method = {RequestMethod.GET})
    @ResponseBody
    public JSONObject formData(Address address, Model model, HttpServletRequest request) {
        Long accountsId = UserUtils.getmpaccounts(request);
        JSONObject rtnJson = new JSONObject();
        try {
            if (address != null && address.getId() != null) {
                address.setUpdateBy(UserUtils.getUser().getId());
                addressService.update(address);
                rtnJson.put("errorNo", 0);
            } else {
                address.setAccountsId(accountsId);
                address.setDelFlag(false);
                address.setAccountsId(UserUtils.getmpaccounts(request));
                address.setCreateBy(UserUtils.getUser().getId());
                address.setUpdateBy(UserUtils.getUser().getId());
                addressService.save(address);
                rtnJson.put("errorNo", 2);
            }
        } catch (Exception e) {
            rtnJson.put("errorNo", 1);
        }
        return rtnJson;
    }

    //获取列表数据
    @RequiresPermissions(value = ADDRESS_VIEW)
    @RequestMapping(value = "/listData")
    @ResponseBody
    public JSONObject listData(int iDisplayStart, int iDisplayLength, Address address, HttpServletRequest request) {
        Long accountsId = UserUtils.getmpaccounts(request);
        Conds conds = new Conds();
        conds.equal("t.del_flag", Constants.DEL_FLAG_0);
        conds.equal("t.accounts_id", accountsId);
        if (!StringUtils.isEmpty(address.getAddressName())){
            conds.like("t.address_name",address.getAddressName());
        }
        if (!StringUtils.isEmpty(address.getProvinceId())){
            conds.equal("t.province_id",address.getProvinceId());
        }
        if (!StringUtils.isEmpty(address.getCityId())){
            conds.equal("t.city_id",address.getCityId());
        }
        if (!StringUtils.isEmpty(address.getAreaId())){
            conds.equal("t.area_id",address.getAreaId());
        }
        Sort sort = new Sort("t.create_time", OrderType.DESC);

        List<Address> list = addressService.fetchSearchByPage(conds, sort, iDisplayStart, iDisplayLength);
        Integer counts = addressService.count(conds);
        JSONObject rtnJson = new JSONObject();
        rtnJson.put("iTotalRecords", counts);
        rtnJson.put("iTotalDisplayRecords", counts);
        rtnJson.put("aaData", list);
        return rtnJson;
    }

    //删除数据
    @RequiresPermissions(value = ADDRESS_EDIT)
    @RequestMapping("/delete")
    @ResponseBody
    public JSONObject delete(String id) {
        JSONObject rtnJosn = new JSONObject();
        try {
            addressService.delete(id);
            rtnJosn.put("errorNo", 0);
        } catch (Exception e) {
            rtnJosn.put("errorNo", 1);
        }
        return rtnJosn;
    }
    @GetMapping("/listArea")
    @ResponseBody
    public List<Area> listArea(){

        return addressService.listArea();
    }
    @RequestMapping("/exportExcel")
    @RequiresPermissions(value = ADDRESS_VIEW)
    public void exportExcel(HttpServletRequest request, HttpServletResponse response) {
        Long accountId = UserUtils.getmpaccounts(request);
        Sort sort = new Sort("create_time", OrderType.ASC);
        Conds conds = new Conds();
        conds.equal("del_flag", 0);
        conds.equal("accounts_id", accountId);
        List<Address> list = addressService.fetchSearchByPage(conds, sort, 0, 0);
        ExcelKit.$Export(Address.class, response).toExcel(list, "收货地址管理信息");
    }

    @RequestMapping("/templateExcel")
    @RequiresPermissions(value = ADDRESS_VIEW)
    public void templateExcel(HttpServletRequest request, HttpServletResponse response) {
        ExcelKit.$Export(Address.class, response).toExcel(null, "收货地址管理信息");
    }

    @RequestMapping("/importExcel")
    @RequiresPermissions(value = ADDRESS_EDIT)
    public String importExcel(HttpServletRequest request, MultipartFile file, Model model) {
        Long accountId = UserUtils.getmpaccounts(request);
        if (file == null) {
            return list(request, model);
        }
// 构造临时路径来存储上传的文件
        String uploadPath = System.getProperty("java.io.tmpdir");
        File uploadDir = new File(uploadPath);
        if (!uploadDir.exists()) {
            uploadDir.mkdir();
        }
        String fileName = file.getOriginalFilename();
        String filePath = uploadPath + File.separator + fileName;
        File storeFile = new File(filePath);
        try {
            file.transferTo(storeFile);
            ExcelKit.$Import().setEmptyCellValue("").readExcel(storeFile, rowData -> {
                if (!StringUtils.isEmpty(rowData.get(0))) {
                    Address address = new Address();
                    address.setAccountsId(accountId);
                    addressService.save(address);
                }
            });
        } catch (IOException e) {
            log.error(e.getMessage());
        }

        return list(request, model);
    }

}
