package com.cftech.orderdetail.web;

import com.alibaba.fastjson.JSONObject;
import com.cftech.orderdetail.model.OrderDetails;
import com.cftech.orderdetail.service.OrderDetailsService;
import com.cftech.core.poi.ExcelKit;
import com.cftech.core.scope.OrderType;
import com.cftech.core.sql.Conds;
import com.cftech.core.sql.Sort;
import com.cftech.core.util.Constants;
import com.cftech.sys.security.PermissionSign;
import com.cftech.sys.security.UserUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.shiro.authz.annotation.RequiresPermissions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.File;
import java.io.IOException;
import java.util.List;


/**
 * 订单管理Controller
 * <p>
 * 权限字符串说明:
 * 查看:public static final String ORDERDETAILS_VIEW = "qy:orderDetails:view"
 * 查看:public static final String ORDERDETAILS_EDIT = "qy:orderDetails:edit"
 *
 * @author Licc
 * @date: 2020-10-16 11:26
 */
@Slf4j
@Controller
@RequestMapping("/a/orderDetails")
public class OrderDetailsController {

    public static final String ORDERDETAILS_VIEW = "qy:orderDetails:view";
    public static final String ORDERDETAILS_EDIT = "qy:orderDetails:edit";

    @Autowired
    private OrderDetailsService orderDetailsService;


    //列表页面
    @RequiresPermissions(value = ORDERDETAILS_VIEW)
    @RequestMapping("/list")
    public String list(HttpServletRequest request, Model model) {
        Long accountId = UserUtils.getmpaccounts(request);
        model.addAttribute("accountId", accountId);
        return "orderDetails/orderDetailslist";
    }

    //编辑页面（新增、修改）
    @RequiresPermissions(value = ORDERDETAILS_VIEW)
    @RequestMapping("/form")
    public String form(HttpServletRequest request, String id, Model model) {
        if (!StringUtils.isEmpty(id)) {
            OrderDetails orderDetails = orderDetailsService.fetchById(id);
            JSONObject jsonObject = JSONObject.parseObject(JSONObject.toJSONString(orderDetails));
            model.addAttribute("data", jsonObject);
        }
        return "orderDetails/orderDetailsform";
    }

    //提交数据（新增、修改）
    @RequiresPermissions(value = ORDERDETAILS_EDIT)
    @RequestMapping("/formData")
    @ResponseBody
    public JSONObject formData(OrderDetails orderDetails, Model model, HttpServletRequest request) {
        Long accountsId = UserUtils.getmpaccounts(request);
        JSONObject rtnJson = new JSONObject();
        try {
            if (orderDetails != null && orderDetails.getId() != null) {
                orderDetails.setUpdateBy(UserUtils.getUser().getId());
                orderDetailsService.update(orderDetails);
                rtnJson.put("errorNo", 0);
            } else {
                orderDetails.setAccountsId(accountsId);
                orderDetails.setDelFlag(false);
                orderDetails.setAccountsId(UserUtils.getmpaccounts(request));
                orderDetails.setCreateBy(UserUtils.getUser().getId());
                orderDetails.setUpdateBy(UserUtils.getUser().getId());
                orderDetailsService.save(orderDetails);
                rtnJson.put("errorNo", 2);
            }
        } catch (Exception e) {
            rtnJson.put("errorNo", 1);
        }
        return rtnJson;
    }

    //获取列表数据
    @RequiresPermissions(value = ORDERDETAILS_VIEW)
    @RequestMapping(value = "/listData")
    @ResponseBody
    public JSONObject listData(int iDisplayStart, int iDisplayLength, OrderDetails orderDetails, HttpServletRequest request) {
        Long accountsId = UserUtils.getmpaccounts(request);
        Conds conds = new Conds();
        conds.equal("d.del_flag", Constants.DEL_FLAG_0);
        conds.equal("d.accounts_id", accountsId);
        conds.like("d.order_code",orderDetails.getOrderCode());
        conds.like("d.drugs_code",orderDetails.getDrugsCode());
        Sort sort = new Sort("d.create_time", OrderType.DESC);

        List<OrderDetails> list = orderDetailsService.fetchSearchByPage(conds, sort, iDisplayStart, iDisplayLength);
        Integer counts = orderDetailsService.count(conds);
        JSONObject rtnJson = new JSONObject();
        rtnJson.put("iTotalRecords", counts);
        rtnJson.put("iTotalDisplayRecords", counts);
        rtnJson.put("aaData", list);
        return rtnJson;
    }

    //删除数据
    @RequiresPermissions(value = ORDERDETAILS_EDIT)
    @RequestMapping("/delete")
    @ResponseBody
    public JSONObject delete(String id) {
        JSONObject rtnJosn = new JSONObject();
        try {
            orderDetailsService.delete(id);
            rtnJosn.put("errorNo", 0);
        } catch (Exception e) {
            rtnJosn.put("errorNo", 1);
        }
        return rtnJosn;
    }

    @RequestMapping("/exportExcel")
    @RequiresPermissions(value = ORDERDETAILS_VIEW)
    public void exportExcel(HttpServletRequest request, HttpServletResponse response) {
        Long accountId = UserUtils.getmpaccounts(request);
        Sort sort = new Sort("create_time", OrderType.ASC);
        Conds conds = new Conds();
        conds.equal("del_flag", 0);
        conds.equal("accounts_id", accountId);
        List<OrderDetails> list = orderDetailsService.fetchSearchByPage(conds, sort, 0, 0);
        ExcelKit.$Export(OrderDetails.class, response).toExcel(list, "订单管理信息");
    }

    @RequestMapping("/templateExcel")
    @RequiresPermissions(value = ORDERDETAILS_VIEW)
    public void templateExcel(HttpServletRequest request, HttpServletResponse response) {
        ExcelKit.$Export(OrderDetails.class, response).toExcel(null, "订单管理信息");
    }

    @RequestMapping("/importExcel")
    @RequiresPermissions(value = ORDERDETAILS_EDIT)
    public String importExcel(HttpServletRequest request, MultipartFile file, Model model) {
        Long accountId = UserUtils.getmpaccounts(request);
        if (file == null) {
            return list(request, model);
        }
// 构造临时路径来存储上传的文件
        String uploadPath = System.getProperty("java.io.tmpdir");
        File uploadDir = new File(uploadPath);
        if (!uploadDir.exists()) {
            uploadDir.mkdir();
        }
        String fileName = file.getOriginalFilename();
        String filePath = uploadPath + File.separator + fileName;
        File storeFile = new File(filePath);
        try {
            file.transferTo(storeFile);
            ExcelKit.$Import().setEmptyCellValue("").readExcel(storeFile, rowData -> {
                if (!StringUtils.isEmpty(rowData.get(0))) {
                    OrderDetails orderDetails = new OrderDetails();
                    orderDetails.setAccountsId(accountId);
                    orderDetailsService.save(orderDetails);
                }
            });
        } catch (IOException e) {
            log.error(e.getMessage());
        }

        return list(request, model);
    }

}
