package com.cftech.brand.category.web;

import com.alibaba.fastjson.JSONObject;
import com.cftech.base.codingrule.utils.CodingruleUtils;
import com.cftech.brand.category.model.BrandCategory;
import com.cftech.brand.category.service.BrandCategoryService;
import com.cftech.brand.department.model.BrandDepartment;
import com.cftech.brand.department.service.BrandDepartmentService;
import com.cftech.core.poi.ExcelKit;
import com.cftech.core.scope.OrderType;
import com.cftech.core.sql.Conds;
import com.cftech.core.sql.Sort;
import com.cftech.core.util.Constants;
import com.cftech.sys.model.User;
import com.cftech.sys.security.PermissionSign;
import com.cftech.sys.security.UserUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.shiro.authz.annotation.RequiresPermissions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.File;
import java.io.IOException;
import java.util.List;


/**
 * 分类Controller
 *
 * 权限字符串说明:
 * 查看:public static final String BRANDCATEGORY_VIEW = "qy:brandCategory:view"
 * 查看:public static final String BRANDCATEGORY_EDIT = "qy:brandCategory:edit"
 *
 * @author faye
 * @date 2020/01/14
 */
@Slf4j
@Controller
@RequestMapping("/a/brandCategory")
public class BrandCategoryController {

    public static final String BRANDCATEGORY_VIEW = "qy:brandCategory:view";
    public static final String BRANDCATEGORY_EDIT = "qy:brandCategory:edit";

    @Autowired
    private BrandCategoryService brandCategoryService;
    @Autowired
    private CodingruleUtils codingruleUtils;

    @Autowired
    private BrandDepartmentService brandDepartmentService;

    /**
     * 列表页面
     *
     * @param request
     * @param model
     * @return
     */
    @RequiresPermissions(value = BRANDCATEGORY_VIEW)
    @RequestMapping("/list")
    public String list(HttpServletRequest request, Model model) {
        Long accountId = UserUtils.getmpaccounts(request);
        model.addAttribute("accountId", accountId);
        return "brandCategory/brandCategorylist";
    }

    /**
     * 编辑页面（新增、修改）
     *
     * @param request
     * @param id
     * @param model
     * @return
     */
    @RequiresPermissions(value = BRANDCATEGORY_VIEW)
    @RequestMapping("/form")
    public String form(HttpServletRequest request, String id, Model model) {
        if (!StringUtils.isEmpty(id)) {
            BrandCategory brandCategory = brandCategoryService.fetchById(id);
            model.addAttribute("data", brandCategory);
            if(brandCategory.getDepartmentId()!=null){
                String name=brandDepartmentService.getNamesByIds(brandCategory.getDepartmentId());
                model.addAttribute("departmentName",name);

            }
        }
        return "brandCategory/brandCategoryform";
    }

    /**
     * 提交数据（新增、修改）
     *
     * @param brandCategory
     * @param model
     * @param request
     * @return
     */
    @RequiresPermissions(value = BRANDCATEGORY_EDIT)
    @RequestMapping("/formData")
    @ResponseBody
    public JSONObject formData(BrandCategory brandCategory, Model model, HttpServletRequest request) {
        Long accountsId = UserUtils.getmpaccounts(request);
        JSONObject rtnJson = new JSONObject();
        try {
            brandCategory.setUpdateBy(UserUtils.getUser().getId());
            if (brandCategory != null && brandCategory.getId() != null) {
                brandCategoryService.update(brandCategory);
                rtnJson.put("errorNo", 0);
            } else {
                brandCategory.setNumber(codingruleUtils.getNumber(accountsId, BrandCategory.class.getName()));
                brandCategory.setAccountsId(accountsId);
                brandCategory.setDelFlag(false);
                brandCategory.setAccountsId(UserUtils.getmpaccounts(request));
                brandCategory.setCreateBy(UserUtils.getUser().getId());
                brandCategoryService.save(brandCategory);
                rtnJson.put("errorNo", 2);
            }
        } catch (Exception e) {
            log.error("保存分类异常:"+e.getMessage());
            rtnJson.put("errorNo", 1);
        }
        return rtnJson;
    }

    /**
     * 获取列表数据
     *
     * @param iDisplayStart
     * @param iDisplayLength
     * @param brandCategory
     * @param request
     * @return
     */

    @RequiresPermissions(value = BRANDCATEGORY_VIEW)
    @RequestMapping(value = "/listData")
    @ResponseBody
    public JSONObject listData(int iDisplayStart, int iDisplayLength, BrandCategory brandCategory, HttpServletRequest request) {
        Long accountsId = UserUtils.getmpaccounts(request);
        Conds conds = new Conds();
        conds.equal("t.del_flag", Constants.DEL_FLAG_0);
        conds.equal("t.accounts_id", accountsId);
        if (!StringUtils.isEmpty(brandCategory.getNumber())) {
            conds.like("t.number", brandCategory.getNumber());
        }
        if (!StringUtils.isEmpty(brandCategory.getName())) {
            conds.like("t.name", brandCategory.getName());
        }
        Sort sort = new Sort("t.create_time", OrderType.DESC);

        List<BrandCategory> list = brandCategoryService.fetchSearchByPageVo(conds, sort, iDisplayStart, iDisplayLength);
        Integer counts = brandCategoryService.count(conds);
        JSONObject rtnJson = new JSONObject();
        rtnJson.put("iTotalRecords", counts);
        rtnJson.put("iTotalDisplayRecords", counts);
        rtnJson.put("aaData", list);
        return rtnJson;
    }

    /**
     * 删除数据
     *
     * @param id
     * @return
     */
    @RequiresPermissions(value = BRANDCATEGORY_EDIT)
    @RequestMapping("/delete")
    @ResponseBody
    public JSONObject delete(String id) {
        JSONObject rtnJosn = new JSONObject();
        try {
            brandCategoryService.delete(id);
            rtnJosn.put("errorNo", 0);
        } catch (Exception e) {
            rtnJosn.put("errorNo", 1);
        }
        return rtnJosn;
    }

    /**
     * 导出
     *
     * @param request
     * @param response
     */
    @RequestMapping("/exportExcel")
    @RequiresPermissions(value = BRANDCATEGORY_VIEW)
    public void exportExcel(HttpServletRequest request,BrandCategory brandCategory, HttpServletResponse response) {
        Long accountId = UserUtils.getmpaccounts(request);
        Sort sort = new Sort("t.create_time", OrderType.ASC);
        Conds conds = new Conds();
        conds.equal("t.del_flag", 0);
        conds.equal("t.accounts_id", accountId);
        if (!StringUtils.isEmpty(brandCategory.getNumber())) {
            conds.like("number", brandCategory.getNumber());
        }
        if (!StringUtils.isEmpty(brandCategory.getName())) {
            conds.like("t.name", brandCategory.getName());
        }
        List<BrandCategory> list = brandCategoryService.fetchSearchByPageVo(conds, sort, 0, 0);
        ExcelKit.$Export(BrandCategory.class, response).toExcel(list, "分类信息");
    }

    /**
     * 模板下载
     *
     * @param request
     * @param response
     */
    @RequestMapping("/templateExcel")
    @RequiresPermissions(value = BRANDCATEGORY_VIEW)
    public void templateExcel(HttpServletRequest request, HttpServletResponse response) {
        ExcelKit.$Export(BrandCategory.class, response).toExcel(null, "分类信息");
    }

    /**
     * 导入
     *
     * @param request
     * @param file
     * @param model
     * @return
     */
    @RequestMapping("/importExcel")
    @RequiresPermissions(value = BRANDCATEGORY_EDIT)
    @ResponseBody
    public JSONObject importExcel(HttpServletRequest request, MultipartFile file, Model model) {
        Long accountId = UserUtils.getmpaccounts(request);
        JSONObject retJson = new JSONObject();
        retJson.put("errorNo", "1");
        if (file == null) {
            retJson.put("errorMsg", "未选择导入文件");
            return retJson;
        }
        // 构造临时路径来存储上传的文件
        String uploadPath = System.getProperty("java.io.tmpdir");
        File uploadDir = new File(uploadPath);
        if (!uploadDir.exists()) {
            uploadDir.mkdir();
        }
        String fileName = file.getOriginalFilename();
        String filePath = uploadPath + File.separator + fileName;
        File storeFile = new File(filePath);
        try {
            file.transferTo(storeFile);
            ExcelKit.$Import().setEmptyCellValue("").readExcel(storeFile, rowData -> {
                BrandCategory brandCategory = new BrandCategory();
                brandCategory.setName(rowData.get(1));
                //绑定所属部门
                String departmentNumber=rowData.get(2);
                if(!StringUtils.isEmpty(departmentNumber)){
                  String departmentIds=brandDepartmentService.getIdsByNumbers(departmentNumber);
                    if(departmentIds!=null){
                        brandCategory.setDepartmentId(departmentIds);
                    }
                }

                //判断编码是否为空 按照编码更新数据
                if (!StringUtils.isEmpty(rowData.get(0))) {
                    brandCategory.setNumber(rowData.get(0));
                    Conds conds=new Conds();
                    conds.equal("del_flag","0");
                    conds.equal("number",brandCategory.getNumber());
                    BrandCategory oldData=brandCategoryService.fetchSearchByConds(conds);
                    if(oldData!=null){
                        brandCategory.setId(oldData.getId());
                        brandCategory.setUpdateBy(UserUtils.getUser().getId());
                        brandCategoryService.update(brandCategory);
                    }else{
                        brandCategory.setAccountsId(accountId);
                        brandCategory.setCreateBy(UserUtils.getUser().getId());
                        brandCategoryService.save(brandCategory);
                    }
                }else{
                    brandCategory.setAccountsId(accountId);
                    brandCategory.setNumber(codingruleUtils.getNumber(accountId,BrandCategory.class.getName()));
                    brandCategory.setCreateBy(UserUtils.getUser().getId());
                    brandCategoryService.save(brandCategory);
                }
            });
            retJson.put("errorNo","0");
            retJson.put("errorMsg","导入成功");
        } catch (Exception e) {
            log.error(e.getMessage());
            retJson.put("errorMsg","导入失败:"+e.getMessage());
        }

        return retJson ;
    }

}
