package com.cftech.brand.supplier.web;

import com.alibaba.fastjson.JSONObject;
import com.cftech.base.codingrule.utils.CodingruleUtils;
import com.cftech.brand.supplier.model.BrandSupplier;
import com.cftech.brand.supplier.service.BrandSupplierService;
import com.cftech.core.poi.ExcelKit;
import com.cftech.core.scope.OrderType;
import com.cftech.core.sql.Conds;
import com.cftech.core.sql.Sort;
import com.cftech.core.util.Constants;
import com.cftech.sys.model.User;
import com.cftech.sys.security.PermissionSign;
import com.cftech.sys.security.UserUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.shiro.authz.annotation.RequiresPermissions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.File;
import java.io.IOException;
import java.util.List;


/**
 * 供应商信息Controller
 *
 * 权限字符串说明:
 * 查看:public static final String BRANDSUPPLIER_VIEW = "qy:brandSupplier:view"
 * 查看:public static final String BRANDSUPPLIER_EDIT = "qy:brandSupplier:edit"
 *
 * @author faye
 * @date 2020/01/14
 */
@Slf4j
@Controller
@RequestMapping("/a/brandSupplier")
public class BrandSupplierController {

    public static final String BRANDSUPPLIER_VIEW = "qy:brandSupplier:view";
    public static final String BRANDSUPPLIER_EDIT = "qy:brandSupplier:edit";

    @Autowired
    private BrandSupplierService brandSupplierService;
    @Autowired
    private CodingruleUtils codingruleUtils;


    /**
     * 列表页面
     *
     * @param request
     * @param model
     * @return
     */
    @RequiresPermissions(value = BRANDSUPPLIER_VIEW)
    @RequestMapping("/list")
    public String list(HttpServletRequest request, Model model) {
        Long accountId = UserUtils.getmpaccounts(request);
        model.addAttribute("accountId", accountId);
        return "brandSupplier/brandSupplierlist";
    }

    /**
     * 编辑页面（新增、修改）
     *
     * @param request
     * @param id
     * @param model
     * @return
     */
    @RequiresPermissions(value = BRANDSUPPLIER_VIEW)
    @RequestMapping("/form")
    public String form(HttpServletRequest request, String id, Model model) {
        if (!StringUtils.isEmpty(id)) {
            BrandSupplier brandSupplier = brandSupplierService.fetchById(id);
            model.addAttribute("data", brandSupplier);
        }
        return "brandSupplier/brandSupplierform";
    }

    /**
     * 提交数据（新增、修改）
     *
     * @param brandSupplier
     * @param model
     * @param request
     * @return
     */
    @RequiresPermissions(value = BRANDSUPPLIER_EDIT)
    @RequestMapping("/formData")
    @ResponseBody
    public JSONObject formData(BrandSupplier brandSupplier, Model model, HttpServletRequest request) {
        Long accountsId = UserUtils.getmpaccounts(request);
        JSONObject rtnJson = new JSONObject();
        try {
            brandSupplier.setUpdateBy(UserUtils.getUser().getId());
            if (brandSupplier != null && brandSupplier.getId() != null) {
                brandSupplierService.update(brandSupplier);
                rtnJson.put("errorNo", 0);
            } else {
                brandSupplier.setNumber(codingruleUtils.getNumber(accountsId, BrandSupplier.class.getName()));
                brandSupplier.setAccountsId(accountsId);
                brandSupplier.setDelFlag(false);
                brandSupplier.setAccountsId(UserUtils.getmpaccounts(request));
                brandSupplier.setCreateBy(UserUtils.getUser().getId());
                brandSupplierService.save(brandSupplier);
                rtnJson.put("errorNo", 2);
            }
        } catch (Exception e) {
            log.error("保存供应商信息异常:"+e.getMessage());
            rtnJson.put("errorNo", 1);
        }
        return rtnJson;
    }

    /**
     * 获取列表数据
     *
     * @param iDisplayStart
     * @param iDisplayLength
     * @param brandSupplier
     * @param request
     * @return
     */

    @RequiresPermissions(value = BRANDSUPPLIER_VIEW)
    @RequestMapping(value = "/listData")
    @ResponseBody
    public JSONObject listData(int iDisplayStart, int iDisplayLength, BrandSupplier brandSupplier, HttpServletRequest request) {
        Long accountsId = UserUtils.getmpaccounts(request);
        Conds conds = new Conds();
        conds.equal("del_flag", Constants.DEL_FLAG_0);
        conds.equal("accounts_id", accountsId);
        if (!StringUtils.isEmpty(brandSupplier.getNumber())) {
            conds.like("number", brandSupplier.getNumber());
        }
        if (!StringUtils.isEmpty(brandSupplier.getName())) {
            conds.like("name", brandSupplier.getName());
        }
        if (!StringUtils.isEmpty(brandSupplier.getPersonName())) {
            conds.like("person_name", brandSupplier.getPersonName());
        }
        if (!StringUtils.isEmpty(brandSupplier.getPersonMobile())) {
            conds.like("person_mobile", brandSupplier.getPersonMobile());
        }
        if (!StringUtils.isEmpty(brandSupplier.getClasses())) {
            conds.like("classes", brandSupplier.getClasses());
        }
        Sort sort = new Sort("create_time", OrderType.DESC);

        List<BrandSupplier> list = brandSupplierService.fetchSearchByPage(conds, sort, iDisplayStart, iDisplayLength);
        Integer counts = brandSupplierService.count(conds);
        JSONObject rtnJson = new JSONObject();
        rtnJson.put("iTotalRecords", counts);
        rtnJson.put("iTotalDisplayRecords", counts);
        rtnJson.put("aaData", list);
        return rtnJson;
    }

    /**
     * 删除数据
     *
     * @param id
     * @return
     */
    @RequiresPermissions(value = BRANDSUPPLIER_EDIT)
    @RequestMapping("/delete")
    @ResponseBody
    public JSONObject delete(String id) {
        JSONObject rtnJosn = new JSONObject();
        try {
            brandSupplierService.delete(id);
            rtnJosn.put("errorNo", 0);
        } catch (Exception e) {
            rtnJosn.put("errorNo", 1);
        }
        return rtnJosn;
    }

    /**
     * 导出
     *
     * @param request
     * @param response
     */
    @RequestMapping("/exportExcel")
    @RequiresPermissions(value = BRANDSUPPLIER_VIEW)
    public void exportExcel(HttpServletRequest request,BrandSupplier brandSupplier, HttpServletResponse response) {
        Long accountId = UserUtils.getmpaccounts(request);
        Sort sort = new Sort("create_time", OrderType.ASC);
        Conds conds = new Conds();
        conds.equal("del_flag", 0);
        conds.equal("accounts_id", accountId);
        if (!StringUtils.isEmpty(brandSupplier.getNumber())) {
            conds.like("number", brandSupplier.getNumber());
        }
        if (!StringUtils.isEmpty(brandSupplier.getName())) {
            conds.like("name", brandSupplier.getName());
        }
        List<BrandSupplier> list = brandSupplierService.fetchSearchByPage(conds, sort, 0, 0);
        ExcelKit.$Export(BrandSupplier.class, response).toExcel(list, "品牌申请供应商信息");
    }

    /**
     * 模板下载
     *
     * @param request
     * @param response
     */
    @RequestMapping("/templateExcel")
    @RequiresPermissions(value = BRANDSUPPLIER_VIEW)
    public void templateExcel(HttpServletRequest request, HttpServletResponse response) {
        ExcelKit.$Export(BrandSupplier.class, response).toExcel(null, "品牌申请供应商信息");
    }

    /**
     * 导入
     *
     * @param request
     * @param file
     * @param model
     * @return
     */
    @RequestMapping("/importExcel")
    @RequiresPermissions(value = BRANDSUPPLIER_EDIT)
    @ResponseBody
    public JSONObject importExcel(HttpServletRequest request, MultipartFile file, Model model) {
        Long accountId = UserUtils.getmpaccounts(request);
        JSONObject retJson = new JSONObject();
        retJson.put("errorNo", "1");
        if (file == null) {
            retJson.put("errorMsg", "未选择导入文件");
            return retJson;
        }
        Long user=UserUtils.getUser().getId();
        // 构造临时路径来存储上传的文件
        String uploadPath = System.getProperty("java.io.tmpdir");
        File uploadDir = new File(uploadPath);
        if (!uploadDir.exists()) {
            uploadDir.mkdir();
        }
        String fileName = file.getOriginalFilename();
        String filePath = uploadPath + File.separator + fileName;
        File storeFile = new File(filePath);
        try {
            file.transferTo(storeFile);
            ExcelKit.$Import().setEmptyCellValue("").readExcel(storeFile, rowData -> {
                BrandSupplier brandSupplier = new BrandSupplier();
                brandSupplier.setAccountsId(accountId);
                brandSupplier.setName(rowData.get(1));
                brandSupplier.setEmail(rowData.get(2));
                brandSupplier.setPersonName(rowData.get(3));
                brandSupplier.setPersonMobile(rowData.get(4));
                brandSupplier.setClasses(rowData.get(5));
                if (!StringUtils.isEmpty(rowData.get(0))) {
                    brandSupplier.setNumber(rowData.get(0));
                    Conds conds=new Conds();
                    conds.equal("del_flag","0");
                    conds.equal("number",brandSupplier.getNumber());
                    BrandSupplier oldData=brandSupplierService.fetchSearchByConds(conds);
                    if(oldData!=null){
                        brandSupplier.setId(oldData.getId());
                        brandSupplier.setUpdateBy(user);
                        brandSupplierService.update(brandSupplier);
                    }else{
                        brandSupplier.setCreateBy(user);
                        brandSupplierService.save(brandSupplier);
                    }
                }else{
                    brandSupplier.setNumber(codingruleUtils.getNumber(accountId,BrandSupplier.class.getName()));
                    brandSupplier.setCreateBy(user);
                    brandSupplierService.save(brandSupplier);
                }
            });
            retJson.put("errorNo","0");
            retJson.put("errorMsg","导入成功");
        } catch (Exception e) {
            log.error(e.getMessage());
            retJson.put("errorMsg","导入失败:"+e.getMessage());
        }

        return retJson ;
    }

}
