import axios, {Axios, AxiosRequestConfig, AxiosResponse, AxiosResponseHeaders} from 'axios';
import {ElMessage, ElMessageBox} from 'element-plus';
import {getToken} from '@/utils/auth';
import {store} from "@/store";

class LinkRequest {
  public instance: LinkRequestInstance;
  // 拦截器对象
  public interceptorsObj?: RequestInterceptors;

  constructor(config: RequestConfig) {
    this.instance = axios.create(config) as unknown as LinkRequestInstance;
    this.interceptorsObj = config.interceptors;

    this.instance.interceptors.request.use(
      (config: RequestConfig) => {
        // 在发送请求之前做些什么 token
        const token = getToken();
        if (token) {
          config.headers!.common['Authorization'] = `${token}`;
        }
        const grayV = import.meta.env.VITE_API_GRAY_VERSION as string;
        if (grayV) {
          config.headers!.common['GRAY_VERSION'] = grayV;
        }
        return config;
      },
      (error) => {
        // 对请求错误做些什么
        return Promise.reject(error);
      }
    );

    // 使用实例拦截器
    this.instance.interceptors.request.use(
      this.interceptorsObj?.requestInterceptors,
      this.interceptorsObj?.requestInterceptorsCatch,
    );
    this.instance.interceptors.response.use(
      this.interceptorsObj?.responseInterceptors,
      this.interceptorsObj?.responseInterceptorsCatch,
    );

    // 全局响应拦截器保证最后执行
    this.instance.interceptors.response.use<Result>(
      (response: ResponseConfig) => {
        const result = response.data;
        const conf = response.config;
        if (result.code && result.code !== 200) {
          // `token` 过期或者账号已在别处登录
          if (result.code === 408) {
            ElMessageBox.close();
            ElMessageBox.alert('你已被登出，请重新登录', '提示', {}).then(() => {
              store.dispatch('user/logoutFront').then(() => {
                window.location.reload();
              });
            })
          }
          // 需要弹出错误信息
          if (conf.showErrorMsg) {
            if(result.code === 500 || result.code === 401){
              ElMessageBox.close();
              ElMessageBox.alert(result.msg, '提示', {})
            }
          }
          return result;
        } else {
          return result;
        }
      },
      (error) => {
        const conf = error.response.config;
        if (conf.showErrorMsg) {
          // 对响应错误做点什么
          if (error.message.indexOf('timeout') != -1) {
            ElMessage.error('网络超时');
          } else if (error.message == 'Network Error') {
            ElMessage.error('网络连接错误');
          } else {
            if (error.response.data) ElMessage.error(error.response.statusText);
            else ElMessage.error('接口路径找不到');
          }
        }
        return Promise.reject(error);
      }
    );
  }

  request(config: RequestConfig):any {
    return this.instance.request(config);
  }
  getInstance() {
    return this.instance;
  }
}

export default LinkRequest;

export interface LinkRequestInstance extends Axios {
  (config: RequestConfig): Promise<Result>;
  (url: string, config?: RequestConfig): Promise<Result>;
}

export interface RequestInterceptors {
  // 请求拦截
  requestInterceptors?: (config: RequestConfig) => RequestConfig
  requestInterceptorsCatch?: (err: any) => any
  // 响应拦截
  responseInterceptors?: (config: ResponseConfig) => ResponseConfig
  responseInterceptorsCatch?: (err: any) => any
}
// 自定义传入的参数
export interface RequestConfig<D = any> extends AxiosRequestConfig {
  interceptors?: RequestInterceptors,
  data?: D;
  headers?: Record<string, string>;
  // 是否弹出错误信息
  showErrorMsg?: boolean;
}


export interface ResponseConfig<D = any> extends AxiosResponse{
  data: D;
  status: number;
  statusText: string;
  headers: AxiosResponseHeaders;
  config: RequestConfig<D>;
  request?: any;
  msg?: string,
  code?: number
}
