import { Module } from 'vuex';
import {resetRouter} from '@/router';
import { UserState, RootStateTypes } from '@/store/types';
import { getToken, setToken, removeToken, getCacheUserInfo, setCacheUserInfo, removeCacheUserInfo, setAccessToken, removeAccessToken } from '@/utils/auth';
import { login, logout } from '@/api/login/index';

const userModule: Module<UserState, RootStateTypes> = {
  namespaced: true,
  state: {
    token: getToken(),
    userInfo: getCacheUserInfo(),
  },
  mutations: {
    RESET_STATE: (state) => {
      state.token = '';
      state.userInfo = {};
    },
    SET_TOKEN: (state, token: string) => {
      state.token = token
    },
    SET_USER: (state, userInfo: Object) => {
      state.userInfo = userInfo
    },
    UPDATE_USER: (state, userInfo: Object) => {
      state.userInfo = Object.assign(userInfo, state.userInfo)
    },
  },
  actions: {
    login({ commit }, params) {
      return new Promise<void>((resolve, reject) => {
        login(params).then((res: Result) => {
          if(res.code === 200){
            const { data } = res;
            const rootUrl = import.meta.env.VITE_API_URL;
            const userInfo = {
              ...data,
              photo: `${rootUrl}sys/User/getUserPhoto/${data.userId}`,
              time: new Date().getTime()
            };
            delete userInfo.authorization;
            commit('SET_USER', userInfo);
            setCacheUserInfo(userInfo);
            setAccessToken(data.sessionId);
            commit('SET_TOKEN', data.authorization);
            setToken(data.authorization);
            resolve(userInfo);
          } else {
            reject(res.msg)
          }
        }).catch((error: any) => {
          reject(error)
        })
      })
    },
    autoLogin({ commit }, { token }) {
      return new Promise<void>(resolve => {
        commit('SET_TOKEN', token);
        resolve()
      })
    },
    logout({ commit }) {
      return new Promise<void>((resolve, reject) => {
        logout({}).then((res: Result) => {
          if (res.code === 200) {
            removeToken();
            removeCacheUserInfo();
            resetRouter();
            removeAccessToken();
            commit('RESET_STATE');
            resolve()
          } else {
            reject(res.msg)
          }
        }).catch(error => {
          reject(error)
        })
      })
    },

    logoutFront({ commit }) {
      return new Promise<void>((resolve) => {
        removeToken();
        removeCacheUserInfo();
        resetRouter();
        removeAccessToken();
        commit('RESET_STATE');
        resolve()
      })
    }
  },
};

export default userModule;
